#ifndef CELL_H
#define CELL_H

//class Cell;

#include <stdio.h>
#include "theme.h"
#include "car.h"

/** A part of a track. The default behaviour is just the ground, and
    subclasses contain actual track portions like curves and stuff */
class Cell {
 public:
  Cell();
  ~Cell();

  /** EDITION and INIT FUNCTIONS FOLLOW **/

  bool init();

  /* Notify this cell that there isn't (wall=true) or there is
     (wall==false) a connecting track on that side so it should put or
     remove a wall. This will probably only ever be called with values
     for which isConnected would return true. */
  virtual void putWall(int side,bool wall);
  /** Try to set the cell's "side" side (0:east, 1:north, 2:west,
      3:south) to height h and slope s. This is used by the editor */
  virtual void setHeight(int side,float h,float s);

  /* different numbers will make it look different. It's up to the
     themes to decide what to do with this. */
  void setVariant(int variant);

  /* Mark this cell as a save point - when the car is at height zero
  for a couple of seconds it is sent back to the latest visited save
  point, at speed zero. The number is the savepoint id. In a track,
  save points must be travelled sequentially, starting with 1. The
  second parameter tells whether this is the last savepoint of a
  track, i.e. a complete track would be like
  (1,false)->(2,false)->...->(n,true)->(1,false).

  UPDATE: pass 0 as n to mark this as *not* being a savepoint
  */
  void setSavePoint(int n,bool last);

  /** Pass the coordinates in the grid */
  void setCoordinates(int x,int z);

  /** Create (and initialises) a clone of this cell. This is used by
      the editor - clicking into the grid returns a clone of the
      selected button. */
  virtual Cell *clone();

  /** render the cell... It should use the area for which x and z are
      both between 0 and 1 **/

  virtual void render(Theme *theme);

  /** COLLISION & INTERACTION FUNCTIONS **/

  /* Whether there's a track leaving the cell on that side */
  virtual bool isConnected(int side);

  /* normal vector from the floor/track at this point */
  virtual float *normal(float x,float y,float z);

  /** Passing two parameters between 0 and 1, tells the height at that
      point */
  virtual float height(float x,float y,float z);
  /** Whether there is a wall intersecting the given segment. If
      there's one, the normal [x;y;z] vector at that point is
      returned. If there's none, null is returned. 

      For now, you may assume that this is only called when the
      height() function returned a value larger than y1, for
      (x1,y1,z2) */
  virtual float *checkWall(float x0,float y0,float z0,float x1,float y1,float z1);

  /** Notify this cell that the car is going on it. This may affect
      the car or the cell or both */
  virtual void interact(Car *c,float t);

  /* Pass a pointer to an int holding the latest visited save point,
     and one of these three values will be returned:

     0: this is not a save point
     >0: this is a save point
     >1: lastPoint got updated to its id
     3: you just finished a lap */
  int isSavePoint(int *lastPoint);

  /** if you have a preference regarding what angle should the camera
      be facing when the car is in this cell, put it into angle and
      return true. Otherwise return false */
  virtual bool preferredAngle(float *angle);

  /** INPUT OUTPUT FUNCTIONS FOLLOW **/

  /* Write one line to save this cell into a file. Does NOT include
     the terminating linefeed. */
  virtual void writeTo(FILE *f);

  /* Read one line from f to configure this cell. Items 1-3 have
     already been consumed (see track.save()) */
  virtual bool initFrom(FILE *f);

  /* Write items 1-3 of a cell specification, without newline (see track.save()) */
  void prefixTo(FILE *f);

 protected:
  /* Coordinates in the grid */
  int x;
  int z;
  /* different numbers will make it look different. It's up to the
     themes to decide what to do with this. */
  int variant;

  int savePoint;
  bool last;

};

/* dammit, C++, this is 2008. We have compilers that can resolve
   forward references today. */

#include "multicell.h"

/** This function reads a one-line description of a cell, allocates a
   cell of the corresponding type and returns it (is there an
   equivalent to java's "static" methods in c++ ?) 

   If a non-zero multicell is passed, the resulting cell(s) is added
   into its stack and the multicell is returned. */

Cell *readFrom(FILE *f,Multicell *m = 0);


#endif
