#include <math.h>
//#include <unistd.h> // usleep

#include "main.h"
#include "domino.h"

// camera (forward/backward) motion
#define CSLOWDOWN 30000.0f
// "strafe" motion
#define STSLOWDOWN 1000000.0f
// sight rotation
#define SSLOWDOWN 5000.0f
// works backwards - the bigger that number is, the less friction you
// have Must be positive.
#define FRICTION 0.5f

float camera[3];
float speed[3]; // camera position speed
float camaccel[3];
float camjerk[3]; // force of current importance class

float sight[3]; // which way we're looking
float sightspeed[3];
float sightaccel[3];
float sightjerk[3];

int totweight; // how many times lookAt has been called since last normCam()

float sq(float z) {return z*z;}

/* Camera works as follows: 

1. Partition the set of "important points" in the scene, into classes
of equivalent importance. For example the scene corners have low
importance, the moving dominoes have medium importance and the player
has high importance if he's moving. The effect of one class is
unaffected by the number of elements - it's only the importance as
passed to normCam() that matters.

2. Call lookAt() to all points in one importance class (this
   increments the -jerk vectors)

3. Call normCam telling the importance of that class, which has the
   effect of actually accelerating the camera (this passes the -jerk
   vectors into the -accel ones)

4. Go back to 2. until all classes have been covered

5. Call updateCam() to apply the acceleration (this passes -speed to
   -position and the -accel vectors to -speed). */

void lookAt(float x,float y,int visible) {
  totweight ++; //= weight;
  /*** First we try to have (x,y,0) in the visible area by moving
       backwards or forwards ***/

  /* the distance from the camera to the plane perpendicular to
     <sight> and containing (x,y,0) */
  float norm = sight[0]*(x-camera[0])+sight[1]*(y-camera[1])-sight[2]*camera[2];
  /* the distance, on that plane, from the point being looked at to
     the object (x,y,0) */
  float rad = sqrtf
    (sq(x- (camera[0]+norm*sight[0]))+
     sq(y- (camera[1]+norm*sight[1]))+
     sq(   (camera[2]+norm*sight[2])));

  //  camaccel[0] += (x-camera[0])/10000.0f;
  //  camaccel[1] += (y-camera[1])/10000.0f;
  /* sort of radial distance from (x,y,0) to a cone corresponding to
     our point of view (i.e. if something interesting is out of the
     visible area, we'll move backwards - if it's inside it we'll move
     forward - if it's on the border we stay still) */
  float accel = (norm-20-rad*1.5f)/CSLOWDOWN;

  /* if this acceleration makes a hidden object visible then it should
     be strong. If it merely makes an already visible object bigger,
     then decrease it to make it less important */
  if (accel>0) accel /= visible;

  //  printf("camera (%f,%f,%f) sight (%f,%f,%f) norm %f rad %f accel %f\n",camera[0],camera[1],camera[2],sight[0],sight[1],sight[2],norm,rad,accel);

  camjerk[0] += sight[0]*accel;
  camjerk[1] += sight[1]*accel;
  camjerk[2] += sight[2]*accel;

  camjerk[0] += (x-camera[0])/STSLOWDOWN;
  camjerk[1] += (y-camera[1]+10)/STSLOWDOWN;
  //  camaccel[2] += (40-camera[2])/CSLOWDOWN;

  /*** now rotate to try looking directly at (x,y,0) ***/
  /* The distance from the camera to the object */
  rad = sqrt(sq(x-camera[0])+sq(y-camera[1])+sq(camera[2]));

  if (accel>0) rad*=visible; //don't rotate too much if object already visible

  sightjerk[0] += ((x-camera[0])/rad) /SSLOWDOWN;
  sightjerk[1] += ((y-camera[1])/rad) /SSLOWDOWN;
  sightjerk[2] -= (   camera[2] /rad) /SSLOWDOWN;
}

void normCam(int weight) {
  //printf("normcam %d (%d points)\n",weight,totweight);
  if (totweight) {
    //printf("camjerk %f %f %f\n",camjerk[0],camjerk[1],camjerk[2]);
    for (int i=0;i<3;i++) {
      camaccel[i] += (camjerk[i]*weight)/totweight;
      sightaccel[i] += (sightjerk[i]*weight)/totweight;
      camjerk[i] = 0;
      sightjerk[i] = 0;
    }
    totweight=0;
    //printf("camaccel %f %f %f\n",camaccel[0],camaccel[1],camaccel[2]);
    //printf("speed %f %f %f\n",speed[0],speed[1],speed[2]);
  }
}

void updateCam(float dt) {
  float friction=1-dt/FRICTION;
  float norm=0;
  if (friction<0) friction=0;

  for (int i=0;i<3;i++) {
    speed[i] += dt*camaccel[i];
    //      camaccel[i] = 0;
    sightspeed[i] += dt*sightaccel[i];
    sightaccel[i] = 0;

    camera[i] += dt*speed[i];
    speed[i] *= friction;

    sight[i] += sightspeed[i];
    sightspeed[i] *= friction;
    
    norm += sight[i]*sight[i];
  }

  norm = sqrtf(norm);
  for (int i=0;i<3;i++) {
    sight[i] /= norm;
  }

  camaccel[0] = 0;
  camaccel[1] = 0;//.001f;
  camaccel[2] = 0;//.002f;
  totweight = 0;//2;
}

void applyCam() {
  glRotatef(atan2(sight[1],hypot(sight[0],sight[2])) RADIANS,-1,0,0);
  glRotatef((atan2(sight[0],-sight[2])) RADIANS,0,1,0);
  //  printf("atan2(%f %f) = %f\n",sight[0],-sight[2],atan2(sight[0],-sight[2]));

  glTranslatef(-camera[0],-camera[1],-camera[2]);
}

void mainLoop(void)
{
  bool done = false;                                     // is our job done ? not yet !
  float lightPos[] = {0.1f,1.0f,1.0f,0};

  SDL_Event event;
  
  Uint32 last=SDL_GetTicks(); // beginning of current generation
  Uint32 prev=last; // previous frame
  Uint32 now=last;
  Uint32 fadeout=0; // if >0 then set light intensity to fadeout/FADETIME
		    // and decrement it for each passing millisecond.
  float t; // how far we are in current generation (t in [0;1) )

  camera[0] = 50;    camera[1] = 23;    camera[2] = 80;
  speed[0] = 0;      speed[1] = 0;      speed[2] = 0;
  camaccel[0] = 0;   camaccel[1] = 0;   camaccel[2] = 0;
  camjerk[0] = 0;   camjerk[1] = 0;   camjerk[2] = 0;

  sight[0] = 0;      sight[1] = 0;      sight[2] = -1;
  sightspeed[0] = 0; sightspeed[1] = 0; sightspeed[2] = 0;
  sightaccel[0] = 0; sightaccel[1] = 0; sightaccel[2] = 0;  
  sightjerk[0] = 0; sightjerk[1] = 0; sightjerk[2] = 0;  

  totweight = 0;

  while(! done)                                          // as long as our job's not done
    {
      prev=now;
      //      now += 10;
      now=SDL_GetTicks();
      //      printf("%d\n",now-prev);
      if (now-prev>25) now=prev+25; // prevent image "shaking" if the computer is too slow
      while( SDL_PollEvent(& event) )                    // look for events (like keystrokes, resizing etc.)
        {
	  switch ( event.type )                          // what kind of event have we got ?
            {
	    case SDL_QUIT :                                         // if user wishes to quit
	      done=true;
	    case SDL_KEYDOWN :           
	      if (event.key.type==SDL_KEYDOWN) {
		switch (event.key.keysym.sym) {
		case SDLK_LEFT:
		  player.move(-1);
		  break;
		case SDLK_RIGHT:
		  player.move(1);
		  break;
		case SDLK_UP:
		  player.climb(1);
		  break;
		case SDLK_DOWN:
		  player.climb(-1);
		  break;
		case SDLK_SPACE:
		  player.grab();
		  break;
		case ',':
		  player.push(-1);
		  break;
		case '.':
		  player.push(1);
		  break;
		default:
		  break;
		}
		//	      } else {
		//		printf("down but not down\n");
	      }
	      break;
	    case SDL_KEYUP :           
	      if (event.key.type==SDL_KEYUP && 
		  (event.key.keysym.sym==SDLK_LEFT ||
		   event.key.keysym.sym==SDLK_RIGHT ||
		   event.key.keysym.sym==SDLK_UP ||
		   event.key.keysym.sym==SDLK_DOWN)) {
		player.stop();
		//	      } else {
		//		printf("up but not up\n");
	      }
	      break;
	    default:                                    // any other event
	      break;                                  // nothing to do
            }
        }

      /* See if the level has been completed (player dying, and other
	 big things will be taken care of here as well) */
        
      if (levelDone) {
	fadeout += (now-prev);
	if (fadeout>=FADETIME) {
	  char pathname[11];
	  fadeout=0;

	  levelDone = false;
	  delete level;
	  levelnum++;
	  
	  if (snprintf(pathname,11,"levels/%d",levelnum)>=11) {
	    printf("internal inconsistency\n");
	    return;
	  }
	  level = new Level();
	  if (!level->load(pathname)) {
	    printf("Can't open level %d (Did you just finish the game?? :)\n",levelnum);
	    return;
	  }
	  
	  Door *wayIn = level->getWayIn();
	  
	  wayIn->open();
	  
	  player.init(wayIn->getCell());
	}
      }
    
      glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT); // Clear The Screen And The Depth Buffer
      glLoadIdentity();                                   // Reset The Matrix
      
      /* bind texture.. */
      //    glEnable( GL_TEXTURE_2D );
    //    glBindTexture(GL_TEXTURE_2D, tex_id);
    // Dois tre rappel  chaque fois !! Sinon a prend le env_mode 
    // du dernier appel
    //    glTexEnvi( GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_REPLACE );  

      while (now-last>TICK) {
	if (!levelDone) {
	  /* stop moving stuff around when we're fading out */
	  if (! player.update()) {
	    levelnum--; // pretend we just "passed" the previous level
	    levelDone = true;
	  }
	  level->update();
	}
	last += TICK;
      }
      if (levelDone) t=0;
      else t=(now-last)/((float)TICK);

      updateCam(now-prev); // note that the lookAt are actually called
			   // by the render()s so their effect is
      applyCam();          // delayed by one frame. Not too dramatic I
			   // suppose...

      /*      lightPos[0] = cosf(now/1000.0f);
      lightPos[1] = 1;
      lightPos[2] = sinf(now/1000.0f);
      lightPos[3] = 0; */
      glLightfv( GL_LIGHT0,GL_POSITION,lightPos);

      player.render(t);

      level->render(t);

      /* apply fadeout effect */

      if (fadeout>0) {
	glColor4f(0,0,0,(float)fadeout/FADETIME);
	glLoadIdentity();
	glBegin(GL_QUADS);
	glVertex3f(-1,-1,-1);
	glVertex3f(-1,+1,-1);
	glVertex3f(+1,+1,-1);
	glVertex3f(+1,-1,-1);
	glEnd();
      }


//     glBegin (GL_TRIANGLE_FAN);  
//     glNormal3f (0,0,1);
//     glTexCoord2f(0,0);  glVertex3f(-1,-1, 0);
//     glTexCoord2f(1,0);  glVertex3f( 1,-1, 0); 
//     glTexCoord2f(1,1);  glVertex3f( 1, 1, 0);
//     glTexCoord2f(0,1);  glVertex3f(-1, 1, 0);
//     glEnd();
    
    // Swap the backbuffers to the foreground
    SDL_GL_SwapBuffers(); 
    } // while( ! done)
}
