#include "domino.h"
#include "main.h"

Domino::Domino() {}

void Domino::init(Cell *c) {
  angle=0;speed=0;accel=0;
  elev=0;fly=0;broken=false;
  top=0;
  enter(c);
  //  printf("initialising domino %p at (%d,%d)\n",this,c->getX(),c->getY());
}

/* try moving to that direction. Less Informative Method Call
   Ever. DOMINOES DON'T CALL touch() ON OTHER DOMINOES - ONLY
   push() */
bool Domino::touch(int direction) {
  //  if (angle!=0) printf("%d touch %d * %d=
  //  %d\n",cell->getX(),direction,angle,direction*angle);
  switch (direction*angle) {
  case -6:
    return neighbour(-2*direction,top)->push(-direction,5*direction) &&
      neighbour(direction,top)->push(direction,-5*direction);
  case -5: case -4: case -3: case -2: case -1:
    return neighbour(direction,top)->push(direction,-5*direction) &&
      neighbour(direction,1-top)->push(direction,-5*direction);
  case 0: case 1: case 2: case 3:
    return neighbour(direction,top)->push(direction,-4*direction) &&
      neighbour(direction,1-top)->push(direction,-4*direction);
  case 4:
    return neighbour(direction,top)->push(direction,0) &&
      neighbour(direction,1-top)->push(direction,0);
  case 5: case 6:
    return neighbour(direction,top)->push(direction,6*direction) &&
      neighbour(2*direction,top)->push(direction,-5*direction);
  default:
    printf("bad touch direction\n");
    return false;
  }
}

bool Domino::push(int direction,int from) {
  //  printf(" (%d).push(%d,%d) ",cell->getX(),direction,from);
  /*
    Assuming direction==+1.

    There's a domino on our left, going towards the right. If it has
    angle:

    -6 or -5, it doesn't push us.
    -4, -3, -2 or -1, it pushes us at -5.
    0, 1, 2 or 3, it pushes us at -4
    4, it pushes us at 0
    5 or 6, it pushes us at 6 (but 6 should break!)

    Now, there's a domino two steps to our left, going towards the
    right. If it has angle:

    between -6 and 4, it doesn't push us.
    5 or 6, it pushes as at -5.

    Conclusion, for direction==+1, we can be pushed at -5, -4, 0 or 6.

    for direction==-1, swap all signs and left/rights
  */

  switch (direction*from) {
  case -5: return (direction*(angle+speed) > -5);
  case -4: return (direction*(angle+speed) > -4);
  case 0:
    if (direction*angle==0)
      return push0(direction);
    else
      return direction*angle > 0;
  case 6: return false;
  default: printf("bad push direction\n"); return false;
  }
}

bool Domino::push0(int direction) {
  accel=direction;
  return true;
}

Domino *Domino::grab() {
  if ((top==0) && (!broken) && (angle==0) && (speed==0) && (fly==0) && (elev==0)) {
    cell->leave(top);
    return this;
  } else
    return 0;
}

void Domino::enter(Cell *c) {
  this->cell = c;
  c->enter(this,top);
}

void Domino::boom() {
  //  printf("domino(%d,%d).BOOM!",cell->getX(),cell->getY());
  broken = true;
  angle=0;
  speed=0;
}

void Domino::land() {
  speed=0;
}

bool Domino::isDone() {
  //  printf("%d+%d\n",angle,speed);
  /* abs(angle)==6 needed for level 9 at least. Don't like it, though :/ */
  return ((abs(angle)==6) || (speed==0 && fly==0 && angle!=0) || (cell->getDomino(top) != this)) && !broken;
}

void Domino::update() {
  //  printf("(dom_up)");
  if (cell->getDomino(top) != this) return;
  //  printf("visible (%d,%d)(%d+%d).",cell->getX(),cell->getY(),angle,speed);
  /* if we are flying then stabilise in vertical position */
  if (fly==0 || (angle!=0))
    angle += speed;

  //  printf(" (%d,%d): angle %d speed %d.",cell->getX(),cell->getY(),angle,speed);

  /* if we are flying then stabilise in vertical position */
  //  if (angle*speed==4) printf("%d angle~4\n",cell->getX());
  if (speed) {
    //    lookAtMe();
    if (!touch(speed)) {// there's something blocking our progress
      if (speed*angle==4 && touch(-speed))
	/* bounce back */
	speed = -speed;
      else
	speed = 0;
    }
    if (angle*speed==6) land();
  } else if (angle!=0) {
    /*    printf("releasing angle %d (%d,%d)\n",angle,cell->getX(),cell->getY());

    if (angle>0 && angle<6) printf("right:%d\n",touch(1));
    else if (angle<0 && angle>-6) printf("left:%d\n",touch(-1));
    */
    // we're not vertical - gravity makes us push towards sign(angle)
    if (angle>0 && angle<6 && touch(1)) speed=1;
    else if (angle<0 && angle>-6 && touch(-1)) speed=-1;
  }

  /** Now deal with vertical motion */ 
  elev += fly;
  
  /* see if we're over a hole */
  if (elev==0) {
    if (!cell->getFloor()) {
      fly=-1;
      if (speed==0 && angle<0) speed=1;
      else if (speed==0 && angle>0) speed=-1;
      
    } else if (abs(angle)==6 && (speed*angle>=0) && ! level->getCell(cell->getX()+(angle/6),cell->getY())->getFloor()) {
      /* we're flat and our "upper" end is over a hole. Twist and fall */
      speed = angle/6;
      migrate(speed,0);
      angle = -angle;
    }
  } else if (abs(elev)==6) {
    migrate(0,fly);
    elev=0;
    if (cell->getFloor()) fly=0;
  }

  //  if (cell->getX()==6) printf("\n");

}

void Domino::rot(float t) {
  int k=angle*2+speed; // to see which corner is touching the ground
  float a=0;

  /* First translation according to elev, and then only rotation according to angle */

  glTranslatef(0,(elev+fly*t)*CELLHEI / 6.0f,0);

  if ((k==0) || ((fly!=0) && (angle==0))) return;


//   if (abs(angle)==2)
//     b=40;
//   else
//     b=30;

//   if (abs(k)<2)
//     a=angle*b+speed*t*30.0f;
//   else if (abs(k)<4)
//     a=angle*b+speed*t*50.0f;
//   else if (abs(k)<6)
//     a=angle*b+speed*t*10.0f;
//   else
  a=angle*15+speed*t*15.0f;

  if (k>0) {
    glTranslatef(DOMWID,0,0);
    glRotatef(a,0,0,-1);
    glTranslatef(-DOMWID,0,0);
  } else if (k<0) {
    glTranslatef(-DOMWID,0,0);
    glRotatef(a,0,0,-1);
    glTranslatef(DOMWID,0,0);
  }
}

Domino *Domino::neighbour(int dx,int top) {
  return level->getCell(cell->getX()+dx,cell->getY())->getDomino(top);
}

void Domino::migrate (int dx,int dy) {
  cell->leave(top);
  cell = level->getCell(cell->getX()+dx,cell->getY()+dy);
  cell->enter(this,top);
}

void tetra() {
  glBegin(GL_TRIANGLES);
  glVertex3f(-1,0,-1);
  glVertex3f(0,0,-0.8f);
  glVertex3f(-0.1f,1,-0.5f);

  glVertex3f(-0.1f,1,-0.5f);
  glVertex3f(0,0,-0.8f);
  glVertex3f(0.2f,0.1f,0.1f);

  glVertex3f(0.2f,0.1f,0.1f);
  glVertex3f(0,0,-0.8f);
  glVertex3f(-1,0,-1);
  glEnd();
}

bool Domino::renderBroken(float t) {
  /* Putting the look at me only because this is usually called by
  every domino, at every frame. 'cause it doesn't actually matter when
  it's called */
  if (speed||fly) lookAtMe(t);

  if (!broken) return true;

  glPushMatrix();

  glColor3f(1,0.5f,0);
  tetra();

  glTranslatef(1,0,0.3f);

  glColor3f(1,0,0);
  glRotatef(40,-1,0,2);
  tetra();
  glRotatef(-40,-1,0,2);

  glTranslatef(1,0,0.3f);

  glColor3f(1,1,0);
  glRotatef(50,1,0,2);
  tetra();
  glRotatef(-50,1,0,2);

  glTranslatef(1,0,0.3f);
  glRotatef(40,-1,0,2);

  glColor3f(1,0.5f,0);
  tetra();

  glTranslatef(1,0,0.3f);
  glRotatef(40,-1,0,2);

  glColor3f(1,1,0);
  tetra();

  glTranslatef(1,0,0.3f);
  glRotatef(40,-1,0,2);

  glColor3f(1,1,0);
  tetra();

  glTranslatef(1,0,0.3f);
  glRotatef(40,-1,0,2);

  glTranslatef(1,0,0.3f);
  tetra();

  glTranslatef(1,0,0.3f);
  glRotatef(40,-1,0,2);

  glTranslatef(1,0,0.3f);
  tetra();

  glTranslatef(1,0,0.3f);
  glRotatef(40,-1,0,2);

  glTranslatef(1,0,0.3f);
  tetra();

  glTranslatef(1,0,0.3f);
  glRotatef(40,-1,0,2);

  glColor3f(1,0.5f,0);
  tetra();

  glPopMatrix();
  return false;
}

void Domino::lookAtMe(float t) {
  /* We focus horizontally on the middle of the domino, and vertically
     on its bottom end */
  lookAt((cell->getX()+(angle+t*speed)/10.0f)*DOMSPC,cell->getY()*CELLHEI+elev+t*fly,1);
}

void Domino::render(float t) {
  /* process push()es */
  if (accel!=0) speed=accel;
  accel=0;

  glPushMatrix();

  rot(t);

  if (renderBroken(t)) {
    //    glEnable(GL_LIGHTING);
    glBegin(GL_QUADS);
    glColor3f(1,1,0);

    glNormal3f( 0,0, 1); // front
    glVertex3f(-DOMWID,0, 2);
    glVertex3f(-DOMWID,DOMHEI, 2);
    glVertex3f( DOMWID,DOMHEI, 2);
    glVertex3f( DOMWID,0, 2);

    glNormal3f( 1,0, 0); // right
    glVertex3f( DOMWID,0, 2);
    glVertex3f( DOMWID,DOMHEI, 2);
    glVertex3f( DOMWID,DOMHEI,-2);
    glVertex3f( DOMWID,0,-2);

    glNormal3f( 0,0,-1); // back (is this even ever shown?)
    glVertex3f( DOMWID,0,-2);
    glVertex3f( DOMWID,DOMHEI,-2);
    glVertex3f(-DOMWID,DOMHEI,-2);
    glVertex3f(-DOMWID,0,-2);

    glNormal3f(-1,0, 0); // left
    glVertex3f(-DOMWID,0,-2);
    glVertex3f(-DOMWID,DOMHEI,-2);
    glVertex3f(-DOMWID,DOMHEI, 2);
    glVertex3f(-DOMWID,0, 2);

    glNormal3f(0,1,0); // top
    glVertex3f(-DOMWID,DOMHEI,-2);
    glVertex3f( DOMWID,DOMHEI,-2);
    glVertex3f( DOMWID,DOMHEI,2);
    glVertex3f(-DOMWID,DOMHEI,2);

    glNormal3f(0,-1,0); // bottom
    glVertex3f(-DOMWID,0,-2);
    glVertex3f( DOMWID,0,-2);
    glVertex3f( DOMWID,0, 2);
    glVertex3f(-DOMWID,0, 2);

    glEnd();
  }

  //  glDisable(GL_LIGHTING);

  glPopMatrix();
}

Domino::~Domino() {
}


